//
//  KeyCode.swift
//  Handy Harp Client
//
//  Created by KIRA Ryouta on 2025/11/16.
//
/*
 MIT License

 Copyright (c) 2025 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

import CoreGraphics
import Carbon

enum KeyType {
    case string(text: String)
    case shift_p
    case returnKey
    case command_t
    case command_option_keypad7
    case command_option_keypad8
    case command_option_keypad9
    case command_option_keypadplus
    case escape
}

enum KeySequence {
    case single(key: KeyType)
    case multiple([KeySequence])
}

struct KeyCodePair {
    let keyCode: CGKeyCode
    let flags: CGEventFlags
}

extension KeyModifiers {
    var asEventFlag: CGEventFlags {
        var result: CGEventFlags = []
        if (self.contains(.shift)) {
            result.formUnion(CGEventFlags.maskShift)
        }
        if (self.contains(.option)) {
            result.formUnion(CGEventFlags.maskAlternate)
        }
        if (self.contains(.control)) {
            result.formUnion(CGEventFlags.maskControl)
        }
        if (self.contains(.command)) {
            result.formUnion(CGEventFlags.maskCommand)
        }
        return result
    }
}

extension KeyType {
    func asKeyCodePair() -> KeyCodePair? {
        switch self {
        case let .string(text):
            var keyCode: CGKeyCode = 0
            var modifiers: KeyModifiers = KeyModifiers(rawValue: 0)
            if HHC_convertKeyStringToKeyCode(text, &keyCode, &modifiers) {
                return KeyCodePair(keyCode: keyCode, flags: modifiers.asEventFlag)
            }
            return nil
        case .shift_p:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_P), flags: [.maskShift])
        case .returnKey:
            return KeyCodePair(keyCode: CGKeyCode(kVK_Return), flags: [])
        case .command_t:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_T), flags: [.maskCommand])
        case .command_option_keypad7:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_Keypad7), flags: [.maskCommand, .maskAlternate])
        case .command_option_keypad8:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_Keypad8), flags: [.maskCommand, .maskAlternate])
        case .command_option_keypad9:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_Keypad9), flags: [.maskCommand, .maskAlternate])
        case .command_option_keypadplus:
            return KeyCodePair(keyCode: CGKeyCode(kVK_ANSI_KeypadPlus), flags: [.maskCommand, .maskAlternate])
        case .escape:
            return KeyCodePair(keyCode: CGKeyCode(kVK_Escape), flags: [])
        }
    }
}

enum AppleScriptKeySequence {
    case keystroke(key: String, modifiers: KeyModifiers)
    case keycode(code: String, modifiers: KeyModifiers)
    case multiple([AppleScriptKeySequence])         // 複数を順に送る
}

enum KeyCodeSequence {
    case single(keyCode: KeyCodePair)
    case multiple([KeyCodeSequence])
}

extension KeySequence {
    var asKeyCodeSequence: KeyCodeSequence? {
        switch self {
        case let .single(key):
            guard let keyCodePair = key.asKeyCodePair() else {
                return nil
            }
            return .single(keyCode: keyCodePair)
        case let .multiple(keys):
            var sequences: [KeyCodeSequence] = []
            for key in keys {
                guard let single = key.asKeyCodeSequence else {
                    return nil
                }
                sequences.append(single)
            }
            return .multiple(sequences)
        }
    }
}

func sendKeyCommand(key: CGKeyCode, flags: CGEventFlags) {
    guard let src = CGEventSource(stateID: .hidSystemState) else { return }

    let keyDown = CGEvent(keyboardEventSource: src, virtualKey: key, keyDown: true)
    keyDown?.flags = flags
    let keyUp = CGEvent(keyboardEventSource: src, virtualKey: key, keyDown: false)
    keyUp?.flags = flags

    keyDown?.post(tap: .cghidEventTap)
    keyUp?.post(tap: .cghidEventTap)
}

func sendKeySequenceByKeyCode(_ seq: KeyCodeSequence) {
    switch seq {
    case let .single(keyCodePair):
        sendKeyCommand(key: keyCodePair.keyCode, flags: keyCodePair.flags)
    case let .multiple(seq):
        for item in seq {
            sendKeySequenceByKeyCode(item)
        }
    }
}

func sendKeySequence(_ seq: KeySequence) {
    if let keyCodeSequence = seq.asKeyCodeSequence {
        sendKeySequenceByKeyCode(keyCodeSequence)
    }
}
